﻿#pragma once
#include "Direct3DBase.h"
#include "Align16.h"
#include "ConstantBufferLayouts.h"

namespace RTCam {

class RenderTarget;
class DepthStencil;
class VertexShader;
class PixelShader;
class GeometryShader;

class Scene;
class Entity;
class Camera;

class RTRenderer : public Direct3DBase, public Align16
{
public:
	RTRenderer(void);
	~RTRenderer(void);

#pragma region Direct3DBase_Methods
	virtual void CreateDeviceResources() override;
	virtual void CreateWindowSizeDependentResources() override;
	virtual void Render(const shared_ptr<Scene>& scene) override;
	virtual void UpdateForWindowSizeChange() override;
#pragma endregion

	enum UIDirection {
		UILeft,
		UIRight
	};
	void QueueUIText(const char* text, float x, float y, UIDirection anchor = UILeft, float width = 0, float height = 0); // Queues some text to be drawn to the screen at the given position.

	// Called before each frame is rendered, after RTSimulation's RenderUpdate
	void RenderUpdate(const shared_ptr<Scene>& scene, float time, float delta);

	void ToggleLongExposureRendering();

	int m_debugRenderMode;
	bool m_drawUI;

private:

#pragma region Direct3DBase_Methods
	virtual void ReleaseResources() override;
#pragma endregion

	void CreateShadersAndBindInputLayouts();
	void CreateConstantBuffers();
	void CreateFullscreenQuadBuffers();
	void CreateSamplerAndBlendStates();
	void CreateBokehResources();

	void ResetLongExposureResources();

	void InitialPass(const shared_ptr<Scene>& scene);
	void DrawScene(const shared_ptr<Scene>& scene);
	void DrawEntity(const shared_ptr<Entity>& entity);
	void DepthAndBokehPrepass();
	void BokehPass(_In_ ID3D11ShaderResourceView* source);
	void BokehBlurPass(); // Blur the bokeh RTs to reduce aliasing artifacts
	void DoFMergePass(_In_ ID3D11RenderTargetView* destination);
	void OldDoFPass();
	void MotionBlurPass(_In_ ID3D11ShaderResourceView* source, _In_ ID3D11RenderTargetView* destination);
	void AberrationFirstPass(_In_ ID3D11ShaderResourceView* source, _In_ ID3D11RenderTargetView* destination);
	void AberrationSecondPass(_In_ ID3D11ShaderResourceView* source, _In_ ID3D11RenderTargetView* destination);
	void LongExposurePass(_In_ ID3D11ShaderResourceView* source, _In_ ID3D11RenderTargetView* destination);

	void TonemapPass(_In_ ID3D11ShaderResourceView* shaderView, _In_ ID3D11RenderTargetView* output);
	void TonemapGrainPass(_In_ ID3D11ShaderResourceView* shaderView, _In_ ID3D11RenderTargetView* output);
	void BoxBlurPass(const unique_ptr<RenderTarget>& sourceRT, const unique_ptr<RenderTarget>& intermediateRT);
	void UIPass();
	

	// Debug rendering passes
	void CopyPass(_In_ ID3D11ShaderResourceView* shaderView, _In_ ID3D11RenderTargetView* output);
	
	void RenderCoCRadius();
	void RenderDepth();
	void RenderVelocity();

	// Renders a basic fullscreen pass using the specified pixel shader
	struct FullscreenParams {
		// Pixel shader stage
		ID3D11PixelShader* pixelShader;
		ID3D11ShaderResourceView** views;
		size_t numViews;
		ID3D11Buffer** buffers;
		size_t numBuffers;

		// Output merger stage
		ID3D11BlendState* blendState;
		ID3D11RenderTargetView** rtViews;
		size_t numRTViews;
	};
	void FullscreenPass(const FullscreenParams& params);

	void UpdateWorldViewProjectionMatrices(const shared_ptr<Entity>& entity, CXMMATRIX viewProjection, bool resetCameraMotion);

	void DrawUIText(const wchar_t* text, size_t length, float x, float y, float width, float height);

	bool m_deviceReady;
	bool m_cameraActive;
	bool m_cameraWarningPrinted;
	bool m_renderingLongExposure;

	int m_numBokehPoints;
	float m_currentLongExposureDuration;

	// Whether the camera's motion has been reset (e.g. after switching cameras or making the camera teleport)
	bool m_cameraMotionReset;

	D2D1_SIZE_F m_d2dUISize;

	struct UIText {
		wstring text;
		float x;
		float y;
		UIDirection anchor;
		// Set width/height to 0 to have it use all available space (excluding the x/y offset)
		float width;
		float height;
	};
	vector<UIText> m_queuedText;

	// The camera being used for rendering (updated every frame)
	weak_ptr<Camera> m_currentCamera;

	// Framebuffer views for intermediate passes
	unique_ptr<RenderTarget> m_intermediateRT;
	unique_ptr<DepthStencil> m_intermediateDS;
	unique_ptr<RenderTarget> m_depthBokehRadiusRT;
	unique_ptr<RenderTarget> m_velocityMapRT;
	unique_ptr<RenderTarget> m_bokehFgBlurRT;
	unique_ptr<RenderTarget> m_bokehBgBlurRT;
	unique_ptr<RenderTarget> m_bokehBlurTempRT;
	unique_ptr<RenderTarget> m_postprocessRT1;
	unique_ptr<RenderTarget> m_postprocessRT2;
	unique_ptr<RenderTarget> m_longExposureRT;

	// Shaders
	unique_ptr<VertexShader> m_initialVShader;
	unique_ptr<PixelShader> m_initialPShader;
	unique_ptr<PixelShader> m_depthBokehRadiusPShader;
	unique_ptr<VertexShader> m_dofVShader;
	unique_ptr<GeometryShader> m_dofGShader;
	unique_ptr<PixelShader> m_dofPShader;
	unique_ptr<PixelShader> m_dofMergePShader;
	unique_ptr<PixelShader> m_aberrationFirstPassPShader;
	unique_ptr<PixelShader> m_aberrationSecondPassPShader;
	unique_ptr<PixelShader> m_motionBlurPShader;
	unique_ptr<PixelShader> m_longExposurePShader;
	unique_ptr<VertexShader> m_fullscreenVShader;
	unique_ptr<VertexShader> m_fullscreenDebugVShader;
	unique_ptr<PixelShader> m_oldDoFPShader;
	unique_ptr<PixelShader> m_tonemapPShader;
	unique_ptr<PixelShader> m_tonemapGrainPShader;
	unique_ptr<PixelShader> m_boxBlurHorizPShader;
	unique_ptr<PixelShader> m_boxBlurVertPShader;

	// Test shaders
	unique_ptr<VertexShader> m_testVShader;
	unique_ptr<PixelShader> m_testPShader;
	unique_ptr<PixelShader> m_CoCPShader;
	unique_ptr<PixelShader> m_depthPShader;
	unique_ptr<PixelShader> m_velocityPShader;
	unique_ptr<PixelShader> m_copyPShader;

	// Input layouts
	ComPtr<ID3D11InputLayout> m_posNormalColorLayout;
	ComPtr<ID3D11InputLayout> m_posTexLayout;

	// Samplers and blend states
	ComPtr<ID3D11SamplerState> m_borderedLinearSampler; // Linear sampler that returns black when past texture bounds
	ComPtr<ID3D11BlendState> m_blendAdditive; // Additive blend that doesn't modify alpha
	ComPtr<ID3D11BlendState> m_blendNonpremultiplied; // Nonpremultiplied alpha blend that doesn't modify source alpha

	// Geometry buffers
	ComPtr<ID3D11Buffer> m_fullscreenQuadVertBuffer;
	ComPtr<ID3D11Buffer> m_fullscreenQuadIndexBuffer;
	ComPtr<ID3D11Buffer> m_bokehVertBuffer;

	// Constant buffers
	ComPtr<ID3D11Buffer> m_modelCBuffer;
	ComPtr<ID3D11Buffer> m_cameraCBuffer;

	
};

} // end namespace